
/*
-------------------------------------------------------------------------
This file is part of WxWidgetsExtensions library.
-------------------------------------------------------------------------

WxExtLib (WxWidgetsExtensions) library
-----------------------------

COPYRIGHT NOTICE:

WxExtLib library Copyright (c) 2003-2007 Daniel Kps

The WxWidgetsExtensions library and associated documentation files (the
"Software") is provided "AS IS".  The author(s) disclaim all
warranties, expressed or implied, including, without limitation, the
warranties of merchantability and of fitness for any purpose.  The
author(s) assume no liability for direct, indirect, incidental,
special, exemplary, or consequential damages, which may result from
the use of or other dealings in the Software, even if advised of the
possibility of such damage.

Permission is hereby granted, free of charge, to any person obtaining
a copy of this Software, to deal in the Software without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Software,
and to permit persons to whom the Software is furnished to do so,
subject to the following conditions:

 1. The origin of this source code must not be misrepresented.
 2. Altered versions must be plainly marked as such and must not be
    misrepresented as being the original source.
 3. This Copyright notice may not be removed or altered from any 
    source or altered source distribution.

End of WxExtLib library Copyright Notice

-------------------------------------------------------------------------
*/

#ifndef _NO_HEADER_INCLUDE

#include "WxExtLibConfig.h"

#if defined(__GNUG__) && (!defined(__APPLE__)) && (!defined(M_NoPragmaInterface))
#   pragma implementation "ConfigEntry.cpp"
#endif

// #include <wx/config.h>
#include <wx/confbase.h>

#include "ConfigEntry.h"

// for possible HACK (see below)
// #include <wx/fileconf.h>

#endif // _NO_HEADER_INCLUDE

//=========================================================================

wxConfigEntryManager::wxConfigEntryManager()
{
    m_ConfigBase = NULL;
}

wxConfigEntryManager::~wxConfigEntryManager()
{
}

//-------------------------------------------------------------------------

void wxConfigEntryManager::addConfigEntry (wxConfigEntry * ConfigEntry)
{
    m_ConfigEntryArray.Add (ConfigEntry);
    ConfigEntry -> setAssociatedConfig (m_ConfigBase);
}

//-------------------------------------------------------------------------

void wxConfigEntryManager::setAssociatedConfig (wxConfigBase * ConfigBase)
{
    m_ConfigBase = ConfigBase;
}

wxConfigBase * wxConfigEntryManager::getAssociatedConfig ()
{
    return m_ConfigBase;
}


//-------------------------------------------------------------------------

bool wxConfigEntryManager::readAllConfigEntries ()
{
    bool IsOk = true;
    // OutputString: intended to collect messages about invalid entries etc.
    // (not yet used)
    wxString OutputString;

    int ConfigEntryCount = m_ConfigEntryArray.GetCount ();
    for (int Index=0; Index < ConfigEntryCount; ++Index)
      {
        wxString MessageString;
        if (! m_ConfigEntryArray.Item(Index) -> read (MessageString))
          IsOk = FALSE;

        if (! MessageString.IsEmpty())
          OutputString.Append (MessageString);
      }

    return IsOk;
}

bool wxConfigEntryManager::writeAllConfigEntries ()
{
    bool IsOk = true;
    // OutputString: intended to collect messages about invalid entries etc.
    // (not yet used)
    wxString OutputString; 

    int ConfigEntryCount = m_ConfigEntryArray.GetCount ();
    for (int Index=0; Index < ConfigEntryCount; ++Index)
      {
        wxString MessageString;
        if (! m_ConfigEntryArray.Item(Index) -> write (MessageString))
          IsOk = FALSE;

        if (! MessageString.IsEmpty())
          OutputString.Append (MessageString);
      }

    // HACK to get config values actually written if wxConfig is of
    // type wxFileConfig (this class seems to call Flush()
    // only in its destructor) - unfortunately, apprarently
    // type information is not available for wxFileConfig
//  if (m_ConfigBase -> IsKindOf (CLASSINFO (wxFileConfig)))
//    {
//      wxFileConfig * FileConfig = (*wxFileConfig) m_ConfigBase;
//      FileConfig -> Flush ();
//    }

    return IsOk;
}

bool wxConfigEntryManager::setDefaultValues ()
{
    bool IsOk = true;

    int ConfigEntryCount = m_ConfigEntryArray.GetCount ();
    for (int Index=0; Index < ConfigEntryCount; ++Index)
      {
        m_ConfigEntryArray.Item(Index) -> setToDefault ();
      }

    return IsOk;
}

//=========================================================================

wxConfigEntry::wxConfigEntry ()
{
    m_IsDontReread = FALSE;
    m_IsDontWrite = FALSE;
    m_IsForcedWrite = FALSE;

    m_IsAlreadyRead = FALSE;
    m_IsChangePending = FALSE;
    m_IsSetPending = FALSE;
    m_ConfigBase = NULL;

    m_ReadMinimalTimeInterval.setMinimalTimeInterval (MinRereadIntervalMS);
}

wxConfigEntry::~wxConfigEntry ()
{
}

//-------------------------------------------------------------------------

void wxConfigEntry::setEntryPath (const wxString & EntryPath)
{
    m_EntryPath = EntryPath;
}

void wxConfigEntry::getEntryPath (wxString & EntryPath)
{
    EntryPath = m_EntryPath;
}

void wxConfigEntry::setAssociatedConfig (wxConfigBase * ConfigBase)
{
    m_ConfigBase = ConfigBase;
}

void wxConfigEntry::setDontReread (bool IsDontReread)
{
    m_IsDontReread = IsDontReread;
}

void wxConfigEntry::setDontWrite (bool IsDontWrite)
{
    m_IsDontWrite = IsDontWrite;
}

void wxConfigEntry::setForcedWrite (bool IsForcedWrite)
{
    m_IsForcedWrite = IsForcedWrite;
}

//-------------------------------------------------------------------------

bool wxConfigEntry::readInternal (wxString & String)
{
    setReadState ();
    wxASSERT(m_ConfigBase != NULL); // possibly wxConfigEntry was not initialized
    return m_ConfigBase -> Read (m_EntryPath, & String);
}

bool wxConfigEntry::readInternal (long & Long)
{
    setReadState ();
    wxASSERT(m_ConfigBase != NULL); // possibly wxConfigEntry was not initialized
    return m_ConfigBase -> Read (m_EntryPath, & Long);
}

bool wxConfigEntry::readInternal (double & Double)
{
    setReadState ();
    wxASSERT(m_ConfigBase != NULL); // possibly wxConfigEntry was not initialized
    return m_ConfigBase -> Read (m_EntryPath, & Double);
}

bool wxConfigEntry::readInternal (bool & Boolean)
{
    setReadState ();
    wxASSERT(m_ConfigBase != NULL); // wxConfigEntry possibly not initialized
    return m_ConfigBase -> Read (m_EntryPath, & Boolean);
}

//-------------------------------------------------------------------------

bool wxConfigEntry::writeInternal (const wxString & String)
{
    if (m_IsDontWrite)
      return true;

    if (getIsWriteRequired())
      {
        setWriteState ();
        wxASSERT(m_ConfigBase != NULL); // wxConfigEntry possibly not initialized
        return m_ConfigBase -> Write (m_EntryPath, String);
      }
    else
      {
        setWriteState ();
        wxASSERT(m_ConfigBase != NULL); // wxConfigEntry possibly not initialized
        return true;
      }
}

bool wxConfigEntry::writeInternal (long Long)
{
    if (m_IsDontWrite)
      return true;

    if (getIsWriteRequired())
      {
        setWriteState ();
        wxASSERT(m_ConfigBase != NULL); // wxConfigEntry possibly not initialized
        return m_ConfigBase -> Write (m_EntryPath, Long);
      }
    else
      {
        setWriteState ();
        wxASSERT(m_ConfigBase != NULL); // wxConfigEntry possibly not initialized
        return true;
      }
}

bool wxConfigEntry::writeInternal (double Double)
{
    if (m_IsDontWrite)
      return true;

    if (getIsWriteRequired())
      {
        setWriteState ();
        wxASSERT(m_ConfigBase != NULL); // wxConfigEntry possibly not initialized
        return m_ConfigBase -> Write (m_EntryPath, Double);
      }
    else
      {
        setWriteState ();
        return true;
      }
}

bool wxConfigEntry::writeInternal (bool Boolean)
{
    if (m_IsDontWrite)
      return true;

    if (getIsWriteRequired())
      {
        setWriteState ();
        wxASSERT(m_ConfigBase != NULL); // wxConfigEntry possibly not initialized
        return m_ConfigBase -> Write (m_EntryPath, Boolean);
      }
    else
      {
        setWriteState ();
        return true;
      }
}

//-------------------------------------------------------------------------

void wxConfigEntry::setReadState ()
{
    m_IsAlreadyRead = true;
    m_IsSetPending = FALSE;
    m_IsChangePending = FALSE;
    // m_LastReadTimeMS.Start ();
    m_ReadMinimalTimeInterval.setBegin ();
}

bool wxConfigEntry::getIsReadRequired ()
{
    if (m_IsSetPending)
      return FALSE;

    return ((! m_IsAlreadyRead)
            || ((! m_IsDontReread) 
                && /* m_LastReadTimeMS.Time() >= MinRereadIntervalMS */
                   m_ReadMinimalTimeInterval.check()));
}

void wxConfigEntry::setWriteState ()
{
    m_IsAlreadyRead = true;
    m_IsSetPending = FALSE;
    m_IsChangePending = FALSE;
}

bool wxConfigEntry::getIsWriteRequired ()
{
    return ((m_IsSetPending && m_IsChangePending)
            || m_IsForcedWrite);
}

//=========================================================================

wxIntegerConfigEntry::wxIntegerConfigEntry()
{
    setRangeAndDefault (0, 0, 0);
    m_CurrentValue = 0;
}

void wxIntegerConfigEntry::init (const wxString & EntryPath, int MinInteger,
                                int MaxInteger, int DefaultInteger)
{
    setEntryPath (EntryPath);
    setRangeAndDefault (MinInteger, MaxInteger, DefaultInteger);
    m_CurrentValue = 0;
}

void wxIntegerConfigEntry::setRangeAndDefault (int MinInteger, int MaxInteger, int DefaultInteger)
{
    m_MinInteger = MinInteger;
    m_MaxInteger = MaxInteger;
    m_DefaultInteger = DefaultInteger;
}

void wxIntegerConfigEntry::get (int & Integer)
{
    wxString MessageString;
    read (MessageString);

    Integer = m_CurrentValue;
}

int wxIntegerConfigEntry::get ()
{
    int Integer;
    get (Integer);
    return Integer;
}

void wxIntegerConfigEntry::get (long & Long)
{
    int TempInteger;
    get (TempInteger);
    Long = TempInteger;
}

void wxIntegerConfigEntry::get (unsigned int & UInt)
{
    int TempInteger;
    get (TempInteger);
    UInt = TempInteger;
}

void wxIntegerConfigEntry::get (unsigned long & ULong)
{
    int TempInteger;
    get (TempInteger);
    ULong = TempInteger;
}

void wxIntegerConfigEntry::set (const int & Integer)
{
    m_IsSetPending = true;
    if ((! m_IsChangePending) && (Integer != m_CurrentValue))
      m_IsChangePending = true;

    if (! (m_MinInteger <= Integer && Integer <= m_MaxInteger))
      {
        // NOTE this case could be an indication of an application internal
        // error
        m_CurrentValue = m_DefaultInteger;
      }
    else
      {
        m_CurrentValue = Integer;
      }
}

//-------------------------------------------------------------------------

bool wxIntegerConfigEntry::write (wxString & WXUNUSED(MessageString))
{
    long TempLong = m_CurrentValue;
    return writeInternal (TempLong);
}

bool wxIntegerConfigEntry::read (wxString & WXUNUSED(MessageString))
{
    long Long = m_CurrentValue;
    if (getIsReadRequired())
      {
        if (readInternal (Long))
          {
            int TempInteger = Long;
            if (m_MinInteger <= TempInteger && TempInteger <= m_MaxInteger)
              {
                m_CurrentValue = TempInteger;
              }
            else
              {
                m_CurrentValue = m_DefaultInteger;
              }
          }
        else
          {
            m_CurrentValue = m_DefaultInteger;
            m_IsSetPending = true;
            m_IsChangePending = true;
          }
        return true;
      }
    else
      return FALSE;
}

void wxIntegerConfigEntry::setToDefault ()
{
    set (m_DefaultInteger);
}

//=========================================================================

wxBooleanConfigEntry::wxBooleanConfigEntry ()
{
    setDefault (FALSE);
    m_CurrentValue = FALSE;
}

void wxBooleanConfigEntry::init (const wxString & EntryPath, bool DefaultBoolean)
{
    setEntryPath (EntryPath);
    setDefault (DefaultBoolean);
    m_CurrentValue = FALSE;
}

void wxBooleanConfigEntry::setDefault (bool DefaultBoolean)
{
    m_DefaultBoolean = DefaultBoolean;
}

void wxBooleanConfigEntry::get (bool & Boolean)
{
    wxString MessageString;
    read (MessageString);

    Boolean = m_CurrentValue;
}

bool wxBooleanConfigEntry::get ()
{
    bool Boolean;
    get (Boolean);
    return Boolean;
}

void wxBooleanConfigEntry::set (const bool & Boolean)
{
    m_IsSetPending = true;
    if ((! m_IsChangePending) && (Boolean != m_CurrentValue))
      m_IsChangePending = true;

    m_CurrentValue = Boolean;
}

#if defined(__WXMSW__)
void wxBooleanConfigEntry::get (BOOL & Boolean)
{
    bool TempBoolean;
    get (TempBoolean);
    Boolean = TempBoolean;
}

void wxBooleanConfigEntry::set (const BOOL & Boolean)
{
    bool TempBoolean = ((Boolean) ? true : FALSE);
    set (TempBoolean);
}
#endif

//-------------------------------------------------------------------------

bool wxBooleanConfigEntry::write (wxString & WXUNUSED(MessageString))
{
    return writeInternal (m_CurrentValue);
}

bool wxBooleanConfigEntry::read (wxString & WXUNUSED(MessageString))
{
    bool TempBoolean = m_CurrentValue;
    if (getIsReadRequired())
      {
        if (readInternal (TempBoolean))
          {
            m_CurrentValue = TempBoolean;
          }
        else
          {
            m_CurrentValue = m_DefaultBoolean;
            m_IsSetPending = true;
            m_IsChangePending = true;
          }
        return true;
      }
    else
      return FALSE;
}

void wxBooleanConfigEntry::setToDefault ()
{
    set (m_DefaultBoolean);
}

//=========================================================================

wxStringConfigEntry::wxStringConfigEntry ()
{
    setDefault ("");
    m_CurrentValue = "";
}

void wxStringConfigEntry::init (const wxString & EntryPath, const wxString & DefaultString)
{
    setEntryPath (EntryPath);
    setDefault (DefaultString);
    m_CurrentValue = "";
}

void wxStringConfigEntry::setDefault (const wxString & DefaultString)
{
    m_DefaultString = DefaultString;
}

void wxStringConfigEntry::get (wxString & String)
{
    wxString MessageString;
    read (MessageString);

    String = m_CurrentValue;
}

wxString wxStringConfigEntry::get ()
{
    wxString String;
    get (String);
    return String;
}

void wxStringConfigEntry::set (const wxString & String)
{
    m_IsSetPending = true;
    if ((! m_IsChangePending) && (String != m_CurrentValue))
      m_IsChangePending = true;

    m_CurrentValue = String;
}

//-------------------------------------------------------------------------

bool wxStringConfigEntry::write (wxString & WXUNUSED(MessageString))
{
    return writeInternal (m_CurrentValue);
}

bool wxStringConfigEntry::read (wxString & WXUNUSED(MessageString))
{
    wxString TempString = m_CurrentValue;
    if (getIsReadRequired())
      {
        if (readInternal (TempString))
          {
            m_CurrentValue = TempString;
          }
        else
          {
            m_CurrentValue = m_DefaultString;
            m_IsSetPending = true;
            m_IsChangePending = true;
          }
        return true;
      }
    else
      return FALSE;
}

void wxStringConfigEntry::setToDefault ()
{
    set (m_DefaultString);
}

//=========================================================================

wxDoubleConfigEntry::wxDoubleConfigEntry()
{
    setRangeAndDefault (0., 0., 0.);
    m_CurrentValue = 0.;
}

void wxDoubleConfigEntry::init (const wxString & EntryPath, double MinDouble,
                                double MaxDouble, double DefaultDouble)
{
    setEntryPath (EntryPath);
    setRangeAndDefault (MinDouble, MaxDouble, DefaultDouble);
    m_CurrentValue = 0.;
}

void wxDoubleConfigEntry::setRangeAndDefault (double MinDouble, double MaxDouble, double DefaultDouble)
{
    m_MinDouble = MinDouble;
    m_MaxDouble = MaxDouble;
    m_DefaultDouble = DefaultDouble;
}

void wxDoubleConfigEntry::get (double & Double)
{
    wxString MessageString;
    read (MessageString);

    Double = m_CurrentValue;
}

double wxDoubleConfigEntry::get ()
{
    double Double;
    get (Double);
    return Double;
}

void wxDoubleConfigEntry::set (const double & Double)
{
    m_IsSetPending = true;
    if ((! m_IsChangePending) && (Double != m_CurrentValue))
      m_IsChangePending = true;

    if (! (m_MinDouble <= Double && Double <= m_MaxDouble))
      {
        // NOTE this case could be an indication of an application internal
        // error
        m_CurrentValue = m_DefaultDouble;
      }
    else
      {
        m_CurrentValue = Double;
      }
}

//-------------------------------------------------------------------------

bool wxDoubleConfigEntry::write (wxString & WXUNUSED(MessageString))
{
    double TempDouble = m_CurrentValue;
    return writeInternal (TempDouble);
}

bool wxDoubleConfigEntry::read (wxString & WXUNUSED(MessageString))
{
    double Double = m_CurrentValue;
    if (getIsReadRequired())
      {
        if (readInternal (Double))
          {
            double TempDouble = Double;
            if (m_MinDouble <= TempDouble && TempDouble <= m_MaxDouble)
              {
                m_CurrentValue = TempDouble;
              }
            else
              {
                m_CurrentValue = m_DefaultDouble;
              }
          }
        else
          {
            m_CurrentValue = m_DefaultDouble;
            m_IsSetPending = true;
            m_IsChangePending = true;
          }
        return true;
      }
    else
      return FALSE;
}

void wxDoubleConfigEntry::setToDefault ()
{
    set (m_DefaultDouble);
}

//=========================================================================

wxDateTimeConfigEntry::wxDateTimeConfigEntry()
{
    // setRangeAndDefault (0, 0, 0);
    // m_CurrentValue = 0;
}

void wxDateTimeConfigEntry::init (const wxString & EntryPath, 
                                  const wxDateTime & MinDateTime,
                                  const wxDateTime & MaxDateTime,
                                  const wxDateTime & DefaultDateTime)
{
    setEntryPath (EntryPath);
    setRangeAndDefault (MinDateTime, MaxDateTime, DefaultDateTime);
    m_CurrentValue = wxDateTime ();
}

void wxDateTimeConfigEntry::setRangeAndDefault (const wxDateTime & MinDateTime, 
                                                const wxDateTime & MaxDateTime, 
                                                const wxDateTime & DefaultDateTime)
{
    m_MinDateTime = MinDateTime;
    m_MaxDateTime = MaxDateTime;
    m_DefaultDateTime = DefaultDateTime;
}

void wxDateTimeConfigEntry::get (wxDateTime & DateTime)
{
    wxString MessageString;
    read (MessageString);

    DateTime = m_CurrentValue;
}

wxDateTime wxDateTimeConfigEntry::get ()
{
    wxDateTime DateTime;
    get (DateTime);
    return DateTime;
}

void wxDateTimeConfigEntry::set (const wxDateTime & DateTime)
{
    m_IsSetPending = true;
    // (!m_CurrentValue.IsValid())
    if ((! m_IsChangePending) && (DateTime != m_CurrentValue))
      {
        m_IsChangePending = true;
      }

    // if (!((!m_MinDateTime.IsValid()) || (m_MinDateTime <= DateTime) 
    // && ((!m_MaxDateTime.IsValid()) || (DateTime <= m_MaxDateTime))))
    if (0)
      {
        wxASSERT (0);
        // NOTE this case could be an indication of an application internal
        // error
        m_CurrentValue = m_DefaultDateTime;
      }
    else
      {
        m_CurrentValue = DateTime;
      }
}

//-------------------------------------------------------------------------

bool wxDateTimeConfigEntry::write (wxString & WXUNUSED(MessageString))
{
    wxString DateTimeString;
    if (m_CurrentValue.IsValid())
      DateTimeString = m_CurrentValue.FormatISODate() << " " << m_CurrentValue.FormatISOTime();

    return writeInternal (DateTimeString);
}

bool wxDateTimeConfigEntry::read (wxString & WXUNUSED(MessageString))
{
    wxDateTime TempDateTime;
    if (getIsReadRequired())
      {
        if (readDateTime (TempDateTime)
            && TempDateTime.IsValid())
          {
            if (((!m_MinDateTime.IsValid()) || (m_MinDateTime <= TempDateTime))
                && ((!m_MaxDateTime.IsValid()) || (TempDateTime <= m_MaxDateTime)))
              {
                m_CurrentValue = TempDateTime;
              }
            else
              {
                m_CurrentValue = m_DefaultDateTime;
              }
          }
        else
          {
            m_CurrentValue = m_DefaultDateTime;
            m_IsSetPending = true;
            m_IsChangePending = true;
          }
        return true;
      }
    else
      return FALSE;
}

bool wxDateTimeConfigEntry::readDateTime(wxDateTime & DateTime)
{
    wxString ISODateTimeFormatString = "%Y-%m-%d %H:%M:%S";

    wxDateTime TempDateTime;
    wxString TempDateTimeString;

    if (!readInternal (TempDateTimeString))
      return false;

    // TODO handling of parse results could be improved, this should be put into
    // a new parseDateTime() function (similar to existing parseDate())
    // TODO add support for date-only and time-only entries
    // TODO use more strict function parseDateTimeWithFormat()
    if (TempDateTime.ParseFormat (TempDateTimeString, ISODateTimeFormatString) != NULL)
      {
        DateTime = TempDateTime;
        return true;
      }
    else
      {
        return false;
      }
}

void wxDateTimeConfigEntry::setToDefault ()
{
    set (m_DefaultDateTime);
}

//=========================================================================
 
