
/*
-------------------------------------------------------------------------
This file is part of ImageMagickExtensions library.
-------------------------------------------------------------------------

ImageMagickExtensions library 0.0.1
-----------------------------------

COPYRIGHT NOTICE:

ImageMagickExtensions library Copyright (c) 2004 Daniel Kps.

The ImageMagickExtensions library and associated documentation files (the
"Software") is provided "AS IS".  The author(s) disclaim all
warranties, expressed or implied, including, without limitation, the
warranties of merchantability and of fitness for any purpose.  The
author(s) assume no liability for direct, indirect, incidental,
special, exemplary, or consequential damages, which may result from
the use of or other dealings in the Software, even if advised of the
possibility of such damage.

Permission is hereby granted, free of charge, to any person obtaining
a copy of this Software, to deal in the Software without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Software,
and to permit persons to whom the Software is furnished to do so,
subject to the following conditions:

 1. The origin of this source code must not be misrepresented.
 2. Altered versions must be plainly marked as such and must not be
    misrepresented as being the original source.
 3. This Copyright notice may not be removed or altered from any 
    source or altered source distribution.

End of ImageMagickExtensions library Copyright notice

-------------------------------------------------------------------------
*/

/*-----------------------------------------------------------------------*/
/*
 * NOTES
 * - this is C only code, but sometimes "//" is used to start a comment line
 *   (adapt to normal C comments if the used compiler should require it)
 */
/*-----------------------------------------------------------------------*/

/*
  Include declarations.
*/
#include "magick/studio.h"
#if defined(__WINDOWS__) || defined(__CYGWIN__)
#define WIN32_LEAN_AND_MEAN
#define VC_EXTRALEAN
#include <windows.h>
#include "magick/cache.h"
#include "magick/colorspace.h"
#include "magick/draw.h"
#include "magick/error.h"
#include "magick/hashmap.h"
#include "magick/memory_.h"
#include "magick/monitor.h"
#include "magick/string_.h"
#include "magick/token.h"
#include "magick/utility.h"
#include "magick/image.h"
#include "magick/constitute.h"
#include "ImageMagickExtensions.h"

/*-----------------------------------------------------------------------*/

/*
 * GetActualUsedColorCountInDIB():
 * - return numbers of colors used in a DIB palette
 *
 * NOTES
 * - the actually used color count may differ from field
 *   BITMAPINFO::biClrUsed (see notes below)
 */
int GetActualUsedColorCountInDIB (BITMAPINFOHEADER * BitmapInfoHeader)
{
    int ColorsUsed = BitmapInfoHeader -> biClrUsed;
    // (for BI_RGB):
    // biClrUsed == 0 is interpreted differently depending on the
    // bit depth used:
    // - for color depths from 1..8 bpp: if biClrUsed is set to zero
    //   this means that a palette for all possible colors is there
    // - for color depths 16, 24 or 32 bpp: if biClrUsed is set to 
    //   zero this means that no palette is used (but biClrUsed
    //   may be set to non-zero if a palette is there for 
    //   output optimization purposes)
    if (BitmapInfoHeader -> biCompression == BI_RGB
        && ColorsUsed == 0 
        && BitmapInfoHeader -> biBitCount >= 1 
        && BitmapInfoHeader -> biBitCount <= 8)
      ColorsUsed = (1 << BitmapInfoHeader -> biBitCount);

    return ColorsUsed;
}

/*-----------------------------------------------------------------------*/

int GetActualUsedColorCountInDIBHandle (HDIB DIBHandle)
{
  BITMAPINFO
    * BitmapInfo;

  BITMAPINFOHEADER
    * BitmapInfoHeader;

  int
    ActualUsedColorCount;

  // obtain pointer to BITMAPINFO and BITMAPINFOHEADER
  if (DIBHandle == NULL)
    return 0;
  BitmapInfo = (BITMAPINFO *) GlobalLock (/* (HGLOBAL*) */ DIBHandle);
  if (BitmapInfo == NULL)
    return 0;
  BitmapInfoHeader = & BitmapInfo -> bmiHeader;

  ActualUsedColorCount = GetActualUsedColorCountInDIB (BitmapInfoHeader);

  GlobalUnlock (/* (HGLOBAL*) */ DIBHandle);

  return ActualUsedColorCount;
}

/*-----------------------------------------------------------------------*/

/*
 * ExpandDIBPixelRow():
 * - expand palette or 16-Bit (15-Bit, actually) DIBs to 24 Bit DIB
 * - this function only handles a single pixel row with PixelCount pixels
 *
 * NOTES
 * - expansion of 1-Bit images will make the expanded image 32 times 
 *   as big and thus may lead to images consuming lots of memory
 */
int ExpandDIBPixelRow (BITMAPINFOHEADER * BitmapInfoHeader,
                       RGBQUAD * PaletteRGBAArray,
                       unsigned long PixelCount, 
                       BYTE * DIBPixelDataPtr,
                       BYTE * ExpandedPixelDataPtr)
{
  unsigned long
    DIBBitsPerPixel,
    DIBBytesPerPixel,
    DIBPixelsPerByte,
    ExpandedBitsPerPixel,
    ExpandedBytesPerPixel,
    PixelIndex,
    IntraPixelByteIndex,
    PixelByteShift,
    PixelByteMask;

  unsigned char
    PaletteByteIndex;

  DIBBitsPerPixel = BitmapInfoHeader -> biBitCount;
  ExpandedBitsPerPixel = 32;

  DIBBytesPerPixel = DIBBitsPerPixel / 8;
  ExpandedBytesPerPixel = ExpandedBitsPerPixel / 8;

  if (DIBBitsPerPixel == 1
      /* || DIBBitsPerPixel == 2  -- probably not a valid Bit depth for a DIB but would work */
      || DIBBitsPerPixel == 4)
    {
      RGBQUAD PaletteRGBA;
      BYTE PixelByte;

      // the DIB has a palette and more than one pixel are represented in a byte
      DIBPixelsPerByte = 8 / DIBBitsPerPixel;
      PixelByteMask = (1 << DIBBitsPerPixel) - 1;

      for (PixelIndex=0; PixelIndex < PixelCount; ++PixelIndex)
        {
          // PERFORMANCE could avoid modulo and multiplications by incrementally
          // determining these values
          IntraPixelByteIndex = DIBPixelsPerByte - (PixelIndex % DIBPixelsPerByte) - 1;
          PixelByteShift = IntraPixelByteIndex * DIBBitsPerPixel;

          PixelByte = *DIBPixelDataPtr;
          PixelByte >>= PixelByteShift;
          PaletteByteIndex = PixelByte & PixelByteMask;

          PaletteRGBA = PaletteRGBAArray [PaletteByteIndex];

          *(ExpandedPixelDataPtr++) = PaletteRGBA.rgbBlue;
          *(ExpandedPixelDataPtr++) = PaletteRGBA.rgbGreen;
          *(ExpandedPixelDataPtr++) = PaletteRGBA.rgbRed;
          *(ExpandedPixelDataPtr++) = PaletteRGBA.rgbReserved;

          if (IntraPixelByteIndex == 0)
            DIBPixelDataPtr += sizeof(BYTE);
        }
    }
  else if (DIBBitsPerPixel == 8)
    {
      // the DIB has a palette and exactly one pixel is represented in a byte
      for (PixelIndex=0; PixelIndex < PixelCount; ++PixelIndex)
        {
          RGBQUAD PaletteRGBA;
          BYTE PaletteByteIndex;
          
          PaletteByteIndex = *DIBPixelDataPtr;

          PaletteRGBA = PaletteRGBAArray [PaletteByteIndex];

          *(ExpandedPixelDataPtr++) = PaletteRGBA.rgbBlue;
          *(ExpandedPixelDataPtr++) = PaletteRGBA.rgbGreen;
          *(ExpandedPixelDataPtr++) = PaletteRGBA.rgbRed;
          *(ExpandedPixelDataPtr++) = PaletteRGBA.rgbReserved;
          
          DIBPixelDataPtr += DIBBytesPerPixel;
        }
    }
  else if (DIBBitsPerPixel == 16)
    {
      // pixel color values are represented without palette as 16 Bit value
      // (5-5-5: Bits for Blue, Green, Red respectively). Highest order
      // bit is unused normally
      // exceptions:
      // - if 'biCompression' is set to BI_BITFIELDS, the ordering and
      //   number of bits used for the three colors can be specified 
      //   differently by specifying bit masks to be used (they are specified
      //   at the bmiColors memory position where normally can be a palette)
      // - some windows versions support only a subset of possible bit masks
      //   so one could assume that these BI_BITFIELDS variants of 16-Bit 
      //   DIBs are rarely encountered in practice

      // WARN should check if WORD is actually 16 Bit wide
      // on the used compiler/platform

      WORD
        RedMask,
        RedShift,
        GreenMask,
        GreenShift,
        BlueMask,
        BlueShift;

      WORD
        * DIBPixelDoubleByteDataPtr;

      const int FiveBitToEightBitLookupTable [32] = {
        0, 8, 16, 24, 32, 41, 49, 57, 65, 74, 82, 90, 98, 106, 115, 123, 
        131, 139, 148, 156, 164, 172, 180, 189, 197, 205, 213, 222, 230, 238, 246, 255};

      BlueMask = 0x001F;
      BlueShift = 0;
      GreenMask = 0x03E0;
      GreenShift = 5;
      RedMask = 0x7C00;
      RedShift = 10;

      DIBPixelDoubleByteDataPtr = (WORD *) DIBPixelDataPtr;

      for (PixelIndex=0; PixelIndex < PixelCount; ++PixelIndex)
        {
          // WARN should check if WORD is actually 16 Bit wide
          // on the used compiler/platform
          WORD PixelDoubleByte = *DIBPixelDoubleByteDataPtr;

          // convert from 5 to 8 Bit using a lookup table to avoid
          // multiplication with 255 and divison by 31
          *(ExpandedPixelDataPtr++) 
            = FiveBitToEightBitLookupTable [(PixelDoubleByte & BlueMask) >> BlueShift];
          *(ExpandedPixelDataPtr++) 
            = FiveBitToEightBitLookupTable [(PixelDoubleByte & GreenMask) >> GreenShift];
          *(ExpandedPixelDataPtr++) 
            = FiveBitToEightBitLookupTable [(PixelDoubleByte & RedMask) >> RedShift];
          *(ExpandedPixelDataPtr++) = 0; // WARN or 255?
          
          ++DIBPixelDoubleByteDataPtr; // increment to next 16-Bit WORD
        }
    }
  else
    {
      return False;
    }

  return True;
}

/*-----------------------------------------------------------------------*/

/*
 * DIBPaletteToImage()
 * - Transfers color values from a DIB palette to ImageMagick Image: 
 *   the whole palette is put into an image of size 
 *   (DIBColorPaletteEntryCount x 1). 
 * - Returns True if the image has a palette, False if there is
 *   no palette
 *
 * NOTES
 * - for converting the image content of a DIB image to an ImageMagick 
 *   image, see DIBToImage()
 *
 * BUGS
 * - error handling is partially missing
 */
MagickExport int DIBPaletteToImage (HDIB DIBHandle, Image ** PaletteImage,
                                    ExceptionInfo * exceptionInfo)
{
  long
    BitmapInfoHeaderSize,
    DIBColorPaletteEntryCount;

  BITMAPINFO
    * BitmapInfo;

  BITMAPINFOHEADER
    * BitmapInfoHeader;

  RGBQUAD
    * PaletteRGBAArray;

  Image
    * image;

  const char 
    * MapString;

  // obtain pointer to BITMAPINFO and BITMAPINFOHEADER
  if (DIBHandle == NULL)
    return False;
  BitmapInfo = (BITMAPINFO *) GlobalLock (/* (HGLOBAL*) */ DIBHandle);
  if (BitmapInfo == NULL)
    return False;
  BitmapInfoHeader = & BitmapInfo -> bmiHeader;

  // setup sizes
  BitmapInfoHeaderSize = BitmapInfoHeader -> biSize; // should be 40 normally

  // extract needed information from BitmapInfoHeader:
  if (BitmapInfoHeader -> biPlanes != 1
      || BitmapInfoHeader -> biCompression != BI_RGB)
    {
      // currently no support for compressed DIB's 
      // or anything exotic (e.g. plane count != 1) or with compression
      GlobalUnlock (/* (HGLOBAL*) */ DIBHandle);
      return False;
    }
  DIBColorPaletteEntryCount = GetActualUsedColorCountInDIB (BitmapInfoHeader);

  if (DIBColorPaletteEntryCount == 0)
    return False;

  // the entries of a color palette of a DIB always use 32 bit entries
  // (blue-green-red-reserved)
  MapString = "BGRA"; // or maybe RGBO

  PaletteRGBAArray = (RGBQUAD *) (((unsigned char *) BitmapInfo) 
                                  + BitmapInfoHeaderSize);

  // transfer color values from palette to ImageMagick Image: 
  // the whole palette is put into a single row with DIBColorPaletteEntryCount
  // "pixels"
  image = ConstituteImage (DIBColorPaletteEntryCount, 
                           /* Height */ 1, 
                           MapString, 
                           CharPixel, 
                           PaletteRGBAArray,
                           exceptionInfo);

  GlobalUnlock (/* (HGLOBAL*) */ DIBHandle);

  * PaletteImage = image;
  return True;
}

/*-----------------------------------------------------------------------*/

MagickExport int ImportExportColorPaletteIntoDIB (BYTE * ImportExportColorPaletteByteArray, 
                                                  int ImportExportColorPaletteEntryCount,
                                                  HDIB DIBHandle,
                                                  int IsImport)
{
  long
    BitmapInfoHeaderSize,
    DIBColorPaletteEntryCount;

  BITMAPINFO
    * BitmapInfo;

  BITMAPINFOHEADER
    * BitmapInfoHeader;

  RGBQUAD
    * DIBColorPaletteRGBAArray;

  // obtain pointer to BITMAPINFO and BITMAPINFOHEADER
  if (DIBHandle == NULL)
    return False;
  BitmapInfo = (BITMAPINFO *) GlobalLock (/* (HGLOBAL*) */ DIBHandle);
  if (BitmapInfo == NULL)
    return False;
  BitmapInfoHeader = & BitmapInfo -> bmiHeader;

  // setup sizes
  BitmapInfoHeaderSize = BitmapInfoHeader -> biSize; // should be 40 normally

  // extract needed information from BitmapInfoHeader:
  if (BitmapInfoHeader -> biPlanes != 1
      || BitmapInfoHeader -> biCompression != BI_RGB)
    {
      // currently no support for compressed DIB's 
      // or anything exotic (e.g. plane count != 1) or with compression
      GlobalUnlock (/* (HGLOBAL*) */ DIBHandle);
      return False;
    }
  DIBColorPaletteEntryCount = GetActualUsedColorCountInDIB (BitmapInfoHeader);

  if (DIBColorPaletteEntryCount == 0)
    {
      GlobalUnlock (/* (HGLOBAL*) */ DIBHandle);
      return False;
    }

  if (DIBColorPaletteEntryCount != ImportExportColorPaletteEntryCount)
    {
      GlobalUnlock (/* (HGLOBAL*) */ DIBHandle);
      return False;
    }

  DIBColorPaletteRGBAArray = (RGBQUAD *) (((unsigned char *) BitmapInfo) 
                                          + BitmapInfoHeaderSize);

  if (IsImport)
    {
      memcpy (DIBColorPaletteRGBAArray,
              ImportExportColorPaletteByteArray,
              DIBColorPaletteEntryCount * sizeof(RGBQUAD));
    }
  else
    {
      memcpy (ImportExportColorPaletteByteArray,
              DIBColorPaletteRGBAArray,
              DIBColorPaletteEntryCount * sizeof(RGBQUAD));
    }

  GlobalUnlock (/* (HGLOBAL*) */ DIBHandle);

  return True;
}

/*-----------------------------------------------------------------------*/

MagickExport int ImageToDIBPalette (Image* image, HDIB DIBHandle,
                                    ExceptionInfo * exceptionInfo)
{
  long
    BitmapInfoHeaderSize,
    DIBColorPaletteEntryCount;

  BITMAPINFOHEADER
    * BitmapInfoHeader;

  BITMAPINFO
    * BitmapInfo;

  const char
    * MapString;

  RGBQUAD
    * ColorPaletteRGBAArray;

  int IsOk;

  // obtain pointer to BITMAPINFO and BITMAPINFOHEADER
  if (DIBHandle == NULL)
    return False;
  BitmapInfo = (BITMAPINFO *) GlobalLock (/* (HGLOBAL*) */ DIBHandle);
  if (BitmapInfo == NULL)
    return False;
  BitmapInfoHeader = & BitmapInfo -> bmiHeader;

  // read out BitmapInfoHeader and check if compatible 
  // with the provided ImageMagick image
  BitmapInfoHeaderSize = BitmapInfoHeader -> biSize;

  if (BitmapInfoHeader -> biPlanes != 1
      || BitmapInfoHeader -> biCompression != BI_RGB)
    {
      // currently no support for compressed DIB's 
      // or anything exotic (e.g. plane count != 1) or with compression
      // BI_BITFIELDS would be easy to support, though
      GlobalUnlock (/* (HGLOBAL*) */ DIBHandle);
      return False;
    }

  DIBColorPaletteEntryCount = GetActualUsedColorCountInDIB (BitmapInfoHeader);
  if (DIBColorPaletteEntryCount == 0
      || DIBColorPaletteEntryCount != image->columns
      || image->rows != 1)
    {
      GlobalUnlock (/* (HGLOBAL*) */ DIBHandle);
      return False;
    }

  // transfer "pixels" (colors) from image directly into the DIB palette
  MapString = "BGRA"; // or maybe RGBO
  ColorPaletteRGBAArray = (RGBQUAD *) (((unsigned char *) BitmapInfo) 
                                       + BitmapInfoHeaderSize);
  IsOk = ExportImagePixels (image,
                            0,
                            0,
                            image->columns,
                            1,
                            MapString,
                            CharPixel,
                            ColorPaletteRGBAArray,
                            exceptionInfo);
  
  if (IsOk == False)
    {
      GlobalUnlock (/* (HGLOBAL*) */ DIBHandle);
      return False;
    }

  GlobalUnlock (/* (HGLOBAL*) */ DIBHandle);

  return True;
}


/*-----------------------------------------------------------------------*/

/*
 * DIBToImage():
 * - creates an IM Image from a Windows HDIB
 * Notes:
 * - currently, only uncompressed 24 Bit and 32 Bit DIB's are supported 
 *   as input DIB's
*/
MagickExport Image * DIBToImage (HDIB DIBHandle,
                                 ExceptionInfo * exceptionInfo)
{
  unsigned long
    DIBBitsPerPixel,
    ExpandedBitsPerPixel,
    RowIndex,
    Width,
    Height;

  long
    BitmapInfoHeaderSize,
    DIBColorPaletteSize,
    PaddedDIBPixelRowSize,
    DIBPixelDataSize,
    ExpandedPixelRowSize;

  BITMAPINFO
    * BitmapInfo;

  BITMAPINFOHEADER
    * BitmapInfoHeader;

  RGBQUAD
    * PaletteRGBAArray;

  BYTE
    * DIBPixelPtr,
    * ExpandedPixelRowPtr;

  Image
    * image;

  const char 
    * MapString;

  int
    IsOk;

  // obtain pointer to BITMAPINFO and BITMAPINFOHEADER
  if (DIBHandle == NULL)
    return (NULL);
  BitmapInfo = (BITMAPINFO *) GlobalLock (/* (HGLOBAL*) */ DIBHandle);
  if (BitmapInfo == NULL)
    return (NULL);
  BitmapInfoHeader = & BitmapInfo -> bmiHeader;

  // setup sizes
  BitmapInfoHeaderSize = BitmapInfoHeader -> biSize; // should be 40 normally

  // extract needed information from BitmapInfoHeader:
  // some programs set biSizeImage != PaddedDIBPixelRowSize * Height, so
  // this field is not really useful (it is not needed here, anyway)
  DIBPixelDataSize = BitmapInfoHeader -> biSizeImage;
  Width = BitmapInfoHeader -> biWidth;
  Height = BitmapInfoHeader -> biHeight;

  if (BitmapInfoHeader -> biPlanes != 1
      || BitmapInfoHeader -> biCompression != BI_RGB)
    {
      // currently no support for compressed DIB's 
      // or anything exotic (e.g. plane count != 1) or with compression
      // BI_BITFIELDS would be easy to support, though
      GlobalUnlock (/* (HGLOBAL*) */ DIBHandle);
      return (NULL);
    }
  DIBColorPaletteSize = GetActualUsedColorCountInDIB (BitmapInfoHeader) * sizeof(RGBQUAD);
  DIBBitsPerPixel = BitmapInfoHeader -> biBitCount;
  // padding of row sizes: formula first rounds bits to next byte, then to
  // next byte divisible by four
  // PaddedDIBPixelRowSize = ((((Width * DIBBitsPerPixel + 7) / 8) + 3) / 4) * 4;
  PaddedDIBPixelRowSize = ((Width * DIBBitsPerPixel + 31) / 32) * 4;

  // BitmapInfoHeader -> biXPelsPerMeter; // WARN should set corresponding field in image
  // BitmapInfoHeader -> biYPelsPerMeter; // WARN should set corresponding field in image

  if (DIBBitsPerPixel <= 16)
    {
      ExpandedBitsPerPixel = 32;
    }
  else
    {
      ExpandedBitsPerPixel = DIBBitsPerPixel;
    }
  ExpandedPixelRowSize = Width * (ExpandedBitsPerPixel / 8);

  switch (ExpandedBitsPerPixel)
    {
    case 24:
      MapString = "BGR";
      break;
    case 32:
      MapString = "BGRA"; // or maybe RGBO
      break;
    default:
      MapString = NULL;
      break;
    }

  PaletteRGBAArray = (RGBQUAD *) (((unsigned char *) BitmapInfo) 
                                  + BitmapInfoHeaderSize);

  // transfer pixels from DIB to ImageMagick Image:
  DIBPixelPtr = (BYTE *) (((unsigned char *) BitmapInfo) 
                          + BitmapInfoHeaderSize 
                          + DIBColorPaletteSize);

  if (DIBBitsPerPixel == 24
      || DIBBitsPerPixel == 32)
    {
      image = ConstituteImage (Width, Height, MapString, CharPixel, DIBPixelPtr,
                               exceptionInfo);
    }
  else
    {
      image = AllocateImage ((ImageInfo *) NULL);
      if (image == (Image *) NULL)
        return ((Image *) NULL);
      
      image -> columns = Width;
      image -> rows = Height;
      SetImage (image, OpaqueOpacity);

      // WARN should use ImageMagick provided alloc and free
      ExpandedPixelRowPtr = malloc (ExpandedPixelRowSize);

      if (ExpandedPixelRowPtr != NULL)
        {
          for (RowIndex = 0; RowIndex < image -> rows ; ++RowIndex)
            {
              IsOk = ExpandDIBPixelRow (BitmapInfoHeader,
                                        PaletteRGBAArray,
                                        Width, 
                                        DIBPixelPtr,
                                        ExpandedPixelRowPtr);
              if (IsOk == False)
                {
                  DestroyImage(image);
                  image = NULL;
                  break;
                }

              // WARN must check error handling
              IsOk = ImportImagePixels (image, 
                                        0, 
                                        RowIndex, 
                                        Width,
                                        1,
                                        MapString,
                                        CharPixel,
                                        ExpandedPixelRowPtr);
              if (IsOk == False)
                {
                  InheritException (exceptionInfo, &image->exception);
                  DestroyImage(image);
                  image = NULL;
                  break;
                }
    
              DIBPixelPtr += PaddedDIBPixelRowSize;
            }

          // WARN should use ImageMagick provided alloc and free
          free (ExpandedPixelRowPtr);
        }
    }

  GlobalUnlock (/* (HGLOBAL*) */ DIBHandle);

  return image;
}


/*-----------------------------------------------------------------------*/

/*
 * ImageToDIB():
 * - creates a Windows HDIB from an IM Image
 * - Notes:
 *  - currently, DIBBitsPerPixel must either 24 or 32
 */
MagickExport HDIB ImageToDIB (Image* image, int DIBBitsPerPixel,
                              ExceptionInfo * exceptionInfo)
{
  unsigned long
    RowIndex,
    Width,
    Height;

  const PixelPacket
    * PixelPacketPtr;

  HDIB
    DIBHandle;

  long
    BitmapInfoHeaderSize,
    DIBColorPaletteSize,
    DIBPixelDataSize,
    PaddedDIBPixelRowSize;

  BITMAPINFOHEADER
    * BitmapInfoHeader;

  BITMAPINFO
    * BitmapInfo;

  const char
    * MapString;

  BYTE
    * DIBDestPixelDataPtr;

  int IsOk;

  Width = image->columns;
  Height = image->rows;

  // setup sizes
  BitmapInfoHeaderSize = sizeof(BITMAPINFOHEADER);
  DIBColorPaletteSize = 0;
  // padding of row sizes: formula first rounds bits to next byte, then to
  // next byte divisible by four
  // PaddedDIBPixelRowSize = ((((Width * DIBBitsPerPixel + 7) / 8) + 3) / 4) * 4;
  PaddedDIBPixelRowSize = ((Width * DIBBitsPerPixel + 31) / 32) * 4;
  DIBPixelDataSize = PaddedDIBPixelRowSize * Height;

  // WARN don't know which GlobalAlloc flags should be really used here
  DIBHandle = GlobalAlloc (GMEM_MOVEABLE | GMEM_DDESHARE,
                           BitmapInfoHeaderSize 
                           + DIBColorPaletteSize 
                           + DIBPixelDataSize);
  if (DIBHandle == NULL)
    return(NULL);

  BitmapInfo = (BITMAPINFO *) GlobalLock (/* (HGLOBAL*) */ DIBHandle);
  BitmapInfoHeader = & BitmapInfo -> bmiHeader;

  // setup the BitmapInfoHeader:
  BitmapInfoHeader -> biSize = sizeof (BITMAPINFOHEADER);
  BitmapInfoHeader -> biWidth = Width;
  BitmapInfoHeader -> biHeight = Height;
  BitmapInfoHeader -> biPlanes = 1;
  BitmapInfoHeader -> biBitCount = DIBBitsPerPixel;
  BitmapInfoHeader -> biCompression = BI_RGB; // ==0, no compression
  BitmapInfoHeader -> biXPelsPerMeter = 0; // WARN should be set to corresponding field in image
  BitmapInfoHeader -> biYPelsPerMeter = 0; // WARN should be set to corresponding field in image
  BitmapInfoHeader -> biSizeImage = DIBPixelDataSize;
  BitmapInfoHeader -> biClrUsed = 0; // no palette for RGBA image
  BitmapInfoHeader -> biClrImportant = 0; 

  // WARN SetImageColorspace necessary?
  SetImageColorspace(image, RGBColorspace);

  switch (DIBBitsPerPixel)
    {
    case 24:
      MapString = "BGR";
      break;
    case 32:
      MapString = "BGRA"; // or maybe RGBO
      break;
    default:
      MapString = NULL;   break;
    }

  // transfer pixels from ImageMagick Image to DIB:
  DIBDestPixelDataPtr = (BYTE *) (((unsigned char *) BitmapInfo) 
                                  + BitmapInfoHeaderSize 
                                  + DIBColorPaletteSize);

  for (RowIndex = 0; RowIndex < image -> rows ; ++RowIndex)
    {
      PixelPacketPtr = AcquireImagePixels (image, 0, RowIndex, image->columns, 1,
                                           exceptionInfo);
      // WARN must add error handling

      if (QuantumDepth == 8
          && DIBBitsPerPixel == 32)
        {
          /* Form of PixelPacket is identical to RGBQUAD when QuantumDepth==8 */
          CopyMagickMemory ((void*) DIBDestPixelDataPtr, 
                            (const void*) PixelPacketPtr,
                            sizeof(PixelPacket) * image -> columns);
        }
      else
        {
          IsOk = ExportImagePixels (image,
                                    0,
                                    RowIndex,
                                    image->columns,
                                    1,
                                    MapString,
                                    CharPixel,
                                    DIBDestPixelDataPtr,
                                    exceptionInfo);

          if (IsOk == False)
            break;
        }

      DIBDestPixelDataPtr += PaddedDIBPixelRowSize;
    }

  GlobalUnlock (/* (HGLOBAL) */ DIBHandle);

  if (IsOk == False)
    {
      GlobalFree  (DIBHandle);
      DIBHandle = NULL;
    }

  return DIBHandle;
}

#endif

/*-----------------------------------------------------------------------*/

#ifdef M_UseIMUnresolvedExternalSymbolHack

// HACK unresolved external symbol
#if !defined(BuildMagickModules)
void  RegisterARTImage(void)  { }
void  RegisterAVIImage(void)  { }
void  RegisterAVSImage(void)  { }
void  RegisterBMPImage(void)  { }
void  RegisterCAPTIONImage(void)  { }
void  RegisterCINImage(void)  { }
void  RegisterCIPImage(void)  { }
void  RegisterCLIPImage(void)  { }
void  RegisterCLIPBOARDImage(void)  { }
void  RegisterCMYKImage(void)  { }
void  RegisterCUTImage(void)  { }
void  RegisterDCMImage(void)  { }
void  RegisterDIBImage(void)  { }
void  RegisterDPSImage(void)  { }
void  RegisterDPXImage(void)  { }
void  RegisterEMFImage(void)  { }
void  RegisterEPTImage(void)  { }
void  RegisterFAXImage(void)  { }
void  RegisterFITSImage(void)  { }
void  RegisterFPXImage(void)  { }
void  RegisterGIFImage(void)  { }
void  RegisterGRAYImage(void)  { }
void  RegisterGRADIENTImage(void)  { }
void  RegisterHISTOGRAMImage(void)  { }
void  RegisterHTMLImage(void)  { }
void  RegisterICONImage(void)  { }
void  RegisterJBIGImage(void)  { }
void  RegisterJPEGImage(void)  { }
void  RegisterJP2Image(void)  { }
void  RegisterLABELImage(void)  { }
void  RegisterMAGICKImage(void)  { }
void  RegisterMAPImage(void)  { }
void  RegisterMATImage(void)  { }
void  RegisterMATTEImage(void)  { }
void  RegisterMETAImage(void)  { }
void  RegisterMIFFImage(void)  { }
void  RegisterMONOImage(void)  { }
void  RegisterMPCImage(void)  { }
void  RegisterMPEGImage(void)  { }
void  RegisterMPRImage(void)  { }
void  RegisterMSLImage(void)  { }
void  RegisterMTVImage(void)  { }
void  RegisterMVGImage(void)  { }
void  RegisterNULLImage(void)  { }
void  RegisterOTBImage(void)  { }
void  RegisterPALMImage(void)  { }
void  RegisterPATTERNImage(void)  { }
void  RegisterPCDImage(void)  { }
void  RegisterPCLImage(void)  { }
void  RegisterPCXImage(void)  { }
void  RegisterPDBImage(void)  { }
void  RegisterPDFImage(void)  { }
void  RegisterPICTImage(void)  { }
void  RegisterPIXImage(void)  { }
void  RegisterPLASMAImage(void)  { }
void  RegisterPNGImage(void)  { }
void  RegisterPNMImage(void)  { }
void  RegisterPREVIEWImage(void)  { }
void  RegisterPSImage(void)  { }
void  RegisterPS2Image(void)  { }
void  RegisterPS3Image(void)  { }
void  RegisterPSDImage(void)  { }
void  RegisterPWPImage(void)  { }
void  RegisterRAWImage(void)  { }
void  RegisterRGBImage(void)  { }
void  RegisterRLAImage(void)  { }
void  RegisterRLEImage(void)  { }
void  RegisterSCRImage(void)  { }
void  RegisterSCTImage(void)  { }
void  RegisterSFWImage(void)  { }
void  RegisterSGIImage(void)  { }
void  RegisterSTEGANOImage(void)  { }
void  RegisterSUNImage(void)  { }
void  RegisterSVGImage(void)  { }
void  RegisterTGAImage(void)  { }
void  RegisterTIFFImage(void)  { }
void  RegisterTILEImage(void)  { }
void  RegisterTIMImage(void)  { }
void  RegisterTTFImage(void)  { }
void  RegisterTXTImage(void)  { }
void  RegisterUILImage(void)  { }
void  RegisterURLImage(void)  { }
void  RegisterUYVYImage(void)  { }
void  RegisterVICARImage(void)  { }
void  RegisterVIDImage(void)  { }
void  RegisterVIFFImage(void)  { }
void  RegisterWBMPImage(void)  { }
void  RegisterWMFImage(void)  { }
void  RegisterWPGImage(void)  { }
void  RegisterXImage(void)  { }
void  RegisterXBMImage(void)  { }
void  RegisterXCImage(void)  { }
void  RegisterXCFImage(void)  { }
void  RegisterXPMImage(void)  { }
#if defined(_VISUALC_)
void  RegisterXTRNImage(void)  { }
#endif
void  RegisterXWDImage(void)  { }
void  RegisterYUVImage(void)  { }
#endif

/*-----------------------------------------------------------------------*/

// HACK unresolved external symbol
#if !defined(BuildMagickModules)
void  UnregisterARTImage(void)  { }
void  UnregisterAVIImage(void)  { }
void  UnregisterAVSImage(void)  { }
void  UnregisterBMPImage(void)  { }
void  UnregisterCAPTIONImage(void)  { }
void  UnregisterCINImage(void)  { }
void  UnregisterCIPImage(void)  { }
void  UnregisterCLIPImage(void)  { }
void  UnregisterCLIPBOARDImage(void)  { }
void  UnregisterCMYKImage(void)  { }
void  UnregisterCUTImage(void)  { }
void  UnregisterDCMImage(void)  { }
void  UnregisterDIBImage(void)  { }
void  UnregisterDPSImage(void)  { }
void  UnregisterDPXImage(void)  { }
void  UnregisterEMFImage(void)  { }
void  UnregisterEPTImage(void)  { }
void  UnregisterFAXImage(void)  { }
void  UnregisterFITSImage(void)  { }
void  UnregisterFPXImage(void)  { }
void  UnregisterGIFImage(void)  { }
void  UnregisterGRAYImage(void)  { }
void  UnregisterGRADIENTImage(void)  { }
void  UnregisterHISTOGRAMImage(void)  { }
void  UnregisterHTMLImage(void)  { }
void  UnregisterICONImage(void)  { }
void  UnregisterJBIGImage(void)  { }
void  UnregisterJPEGImage(void)  { }
void  UnregisterJP2Image(void)  { }
void  UnregisterLABELImage(void)  { }
void  UnregisterMAGICKImage(void)  { }
void  UnregisterMAPImage(void)  { }
void  UnregisterMATImage(void)  { }
void  UnregisterMATTEImage(void)  { }
void  UnregisterMETAImage(void)  { }
void  UnregisterMIFFImage(void)  { }
void  UnregisterMONOImage(void)  { }
void  UnregisterMPCImage(void)  { }
void  UnregisterMPEGImage(void)  { }
void  UnregisterMPRImage(void)  { }
void  UnregisterMSLImage(void)  { }
void  UnregisterMTVImage(void)  { }
void  UnregisterMVGImage(void)  { }
void  UnregisterNULLImage(void)  { }
void  UnregisterOTBImage(void)  { }
void  UnregisterPALMImage(void)  { }
void  UnregisterPATTERNImage(void)  { }
void  UnregisterPCDImage(void)  { }
void  UnregisterPCLImage(void)  { }
void  UnregisterPCXImage(void)  { }
void  UnregisterPDBImage(void)  { }
void  UnregisterPDFImage(void)  { }
void  UnregisterPICTImage(void)  { }
void  UnregisterPIXImage(void)  { }
void  UnregisterPLASMAImage(void)  { }
void  UnregisterPNGImage(void)  { }
void  UnregisterPNMImage(void)  { }
void  UnregisterPREVIEWImage(void)  { }
void  UnregisterPSImage(void)  { }
void  UnregisterPS2Image(void)  { }
void  UnregisterPS3Image(void)  { }
void  UnregisterPSDImage(void)  { }
void  UnregisterPWPImage(void)  { }
void  UnregisterRAWImage(void)  { }
void  UnregisterRGBImage(void)  { }
void  UnregisterRLAImage(void)  { }
void  UnregisterRLEImage(void)  { }
void  UnregisterSCRImage(void)  { }
void  UnregisterSCTImage(void)  { }
void  UnregisterSFWImage(void)  { }
void  UnregisterSGIImage(void)  { }
void  UnregisterSTEGANOImage(void)  { }
void  UnregisterSUNImage(void)  { }
void  UnregisterSVGImage(void)  { }
void  UnregisterTGAImage(void)  { }
void  UnregisterTIFFImage(void)  { }
void  UnregisterTILEImage(void)  { }
void  UnregisterTIMImage(void)  { }
void  UnregisterTTFImage(void)  { }
void  UnregisterTXTImage(void)  { }
void  UnregisterUILImage(void)  { }
void  UnregisterURLImage(void)  { }
void  UnregisterUYVYImage(void)  { }
void  UnregisterVICARImage(void)  { }
void  UnregisterVIDImage(void)  { }
void  UnregisterVIFFImage(void)  { }
void  UnregisterWBMPImage(void)  { }
void  UnregisterWMFImage(void)  { }
void  UnregisterWPGImage(void)  { }
void  UnregisterXImage(void)  { }
void  UnregisterXBMImage(void)  { }
void  UnregisterXCImage(void)  { }
void  UnregisterXCFImage(void)  { }
void  UnregisterXPMImage(void)  { }
#if defined(_VISUALC_)
void  UnregisterXTRNImage(void)  { }
#endif
void  UnregisterXWDImage(void)  { }
void  UnregisterYUVImage(void)  { }
#endif

/*-----------------------------------------------------------------------*/

// HACK unresolved external symbol
extern unsigned int
AnalyzeImage(Image ** imagep,const int integer,char ** charpp)
{
  return 0;
}

#endif // M_UseIMUnresolvedExternalSymbolHack

/*-----------------------------------------------------------------------*/
